package mcfall.raytracer.objects;

import java.util.HashSet;

import mcfall.math.IncompatibleMatrixException;
import mcfall.math.Matrix;
import mcfall.math.NotInvertibleException;
import mcfall.math.Point;
import mcfall.math.Ray;
import mcfall.raytracer.Camera;

// TODO: Auto-generated Javadoc
/**
 * The Class AbstractBoundingBox.
 */
public abstract class AbstractBoundingBox extends AbstractRectangleExtent {
	
	private static final int X=1,Y=2,Z=3;
	
	/** The bounding box. */
	GenericCube boundingBox;
	
	/** The bb initial transform. */
	Matrix bbInitialTransform;
	
	/** The bb working transform. */
	Matrix bbWorkingTransform = Matrix.createIdentityMatrix(4);
	
	/**
	 * Sets the dimensions.
	 * 
	 * @param center the center
	 * @param scale a vector representing the scaling factor of a GenericCube that fits the object
	 */
	private void setDimensions(Point center, Point scale){
		try {
			scale = scale.add(new Point(.01,.01,.01));//make the bounding box slightly larger than the obj
			Matrix t = Matrix.createTranslationMatrix(center.getX(), center.getY(), center.getZ());
			Matrix s = Matrix.createScalingMatrix(scale.getX(), scale.getY(), scale.getZ());
			Matrix tInverse = t.invert();
			bbInitialTransform = tInverse.postmultiply(s).postmultiply(t);
			boundingBox.transform(bbInitialTransform);
		} catch (NotInvertibleException e) {
			throw new RuntimeException("Failed to invert matrix");
		} catch (IncompatibleMatrixException e) {
			throw new RuntimeException("Failed to multiply matricies");
		}
	}
	
	/**
	 * Instantiates a new abstract bounding box.
	 * 
	 * @param center the center
	 * @param size a vector representing the scaling factor of a GenericCube that fits the object
	 */
	public AbstractBoundingBox(Point center, Point size ) {
		super();
		boundingBox = new GenericCube();
		setDimensions(center,size);
	}
	
	/**
	 * Instantiates a new abstract bounding box.
	 * 
	 * @param name the name
	 * @param center the center
	 * @param size a vector representing the scaling factor of a GenericCube that fits the object
	 */
	public AbstractBoundingBox(String name, Point center, Point size) {
		super(name);
		boundingBox = new GenericCube(name);
		setDimensions(center,size);
	}
	
	/**
	 * In bounding box.
	 * 
	 * @param ray the ray
	 * 
	 * @return true, if successful
	 */
	public boolean inBoundingBox(Ray ray) { 
		//there is some repitition from GenericCube but we do this for the sake of performace...there are notable differences so we can't just refactor without loosing performance
		//the following is from hill p635
		double tIn = Double.NEGATIVE_INFINITY, tOut = Double.POSITIVE_INFINITY;
		double tHit, numer=0, denom=0;
		boolean tInSet=false,tOutSet=false;
		
		try {
			ray = ray.transform(boundingBox.getTransform().invert());
		} catch (NotInvertibleException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		for(int i=0;i<6;i++) {
			switch(i) {
				case 0: numer = 1.0 - ray.getStart().getY();denom = ray.getDirection().getValueAt(Y);break;
				case 1: numer = 1.0 + ray.getStart().getY();denom = -1d*ray.getDirection().getValueAt(Y);break;
				case 2: numer = 1.0 - ray.getStart().getX();denom = ray.getDirection().getValueAt(X);break;
				case 3: numer = 1.0 + ray.getStart().getX();denom = -1d*ray.getDirection().getValueAt(X);break;
				case 4: numer = 1.0 - ray.getStart().getZ();denom = ray.getDirection().getValueAt(Z);break;
				case 5: numer = 1.0 + ray.getStart().getZ();denom = -1d*ray.getDirection().getValueAt(Z);break;
			}
			if(Math.abs(denom) < .00001) {
				 if(numer<0) { continue; }
			} else {
				tHit = numer/denom;
				if(denom > 0) {
					if(tHit<tOut) {
						tOut = tHit;
						tInSet = true;
					}
				} else {
					if(tHit > tIn) {
						tIn = tHit;
						tOutSet=true;
					}
				}
			}
			if(tIn >=tOut) return false;
		}
		if(tInSet && tOutSet) {return true;} //if we have determined both a tIn and tOut and they are in the right order
		return false;
	}
	

	/* (non-Javadoc)
	 * @see mcfall.raytracer.objects.AbstractRectangleExtent#transform(mcfall.math.Matrix)
	 */
	@Override
	public void transform(Matrix transform) {
		try {
			bbWorkingTransform = bbWorkingTransform.postmultiply(transform);
			boundingBox.transform(bbWorkingTransform);//this is most likely wrong
		} catch (IncompatibleMatrixException e) {
			e.printStackTrace();
		/*} catch (NotInvertibleException e) {
			e.printStackTrace();*/
		}
		super.transform(transform);
	}
	
	/* (non-Javadoc)
	 * @see mcfall.raytracer.objects.AbstractRectangleExtent#setUp(mcfall.raytracer.Camera)
	 */
	@Override
	public void setUp(Camera camera) {
		HashSet<Point> critPoints = new HashSet<Point>();
		for(int x = -1;x<=1;x+=2) {
			for(int y = -1;y<=1;y+=2) {
				for(int z = -1;z<=1;z+=2) {
					critPoints.add(new Point(x,y,z));
				}
			}
		}
		this.setUp(camera,critPoints);		
	}
	
}
